﻿using Microsoft.AspNetCore.Mvc.Formatters;
using Microsoft.Net.Http.Headers;
using System;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

namespace VA.PPMS.IWS.Api.Formatters
{
    public class StreamInputFormatter : TextInputFormatter
    {
        public StreamInputFormatter()
        {
            SupportedMediaTypes.Add(MediaTypeHeaderValue.Parse("application/octet-stream"));

            SupportedEncodings.Add(Encoding.UTF8);
            SupportedEncodings.Add(Encoding.Unicode);
        }

        protected override bool CanReadType(Type type)
        {
            return type == typeof(string) && base.CanReadType(type);
        }

        public override bool CanRead(InputFormatterContext context)
        {
            if (context == null) throw new ArgumentNullException(nameof(context));

            var request = context.HttpContext.Request;
            var contentType = request.ContentType;

            return string.IsNullOrEmpty(contentType) || 
                contentType == "application/xml" || 
                contentType == "text/plain" || 
                contentType == "application/octet-stream";
        }

        public override async Task<InputFormatterResult> ReadRequestBodyAsync(InputFormatterContext context, Encoding encoding)
        {
            if (context == null)
            {
                throw new ArgumentNullException(nameof(context));
            }

            if (encoding == null)
            {
                throw new ArgumentNullException(nameof(encoding));
            }

            var request = context.HttpContext.Request;

            using (var reader = new StreamReader(request.Body, encoding))
            {
                try
                {
                    var all = await reader.ReadToEndAsync();
                    return await InputFormatterResult.SuccessAsync(all);
                }
                catch (XmlException)
                {
                    return await InputFormatterResult.FailureAsync();
                }
            }
        }
    }
}